package gov.cms.grouper.snf.lego;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class SnfUtilsTest {

  @Test
  public void testClean() {
    String input = " hello world \t\n";
    String expected = "hello world";
    String actual = SnfUtils.clean(input);
    Assertions.assertEquals(expected, actual);

    input = "  \t\n";
    expected = "NULL";
    actual = SnfUtils.clean(input, expected);
    Assertions.assertEquals(expected, actual);

    input = null;
    expected = "NULL";
    actual = SnfUtils.clean(input, expected);
    Assertions.assertEquals(expected, actual);

    input = null;
    expected = null;
    actual = SnfUtils.clean(input, expected);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testBlank() {
    String input = " hello world \t\n";
    boolean expected = false;
    boolean actual = SnfUtils.isBlank(input);
    Assertions.assertEquals(expected, actual);

    input = "  \t\n";
    expected = true;
    actual = SnfUtils.isBlank(input);
    Assertions.assertEquals(expected, actual);

    input = null;
    expected = true;
    actual = SnfUtils.isBlank(input);
    Assertions.assertEquals(expected, actual);

  }


  @Test
  public void testIfElse() {
    Map<String, Boolean> info = new HashMap<>();
    final String key = "key";
    final int random = Integer.MIN_VALUE;
    Supplier<Integer> trueExec = () -> {
      info.put(key, Boolean.TRUE);
      return random;
    };

    Supplier<Integer> falseExec = () -> {
      info.put(key, Boolean.FALSE);
      return random;
    };

    boolean expected = true;
    SnfUtils.ifElse(() -> true, trueExec, falseExec);
    boolean actual = info.get(key);
    Assertions.assertEquals(expected, actual);

    expected = false;
    SnfUtils.ifElse(() -> false, trueExec, falseExec);
    actual = info.get(key);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testNullCheck() {
    String input = " hello world \t\n";
    String expectedS = "good";
    String actualS = SnfUtils.nullCheck(input, null, expectedS);
    Assertions.assertEquals(expectedS, actualS);

    input = null;
    expectedS = "good";
    actualS = SnfUtils.nullCheck(input, expectedS, null);
    Assertions.assertEquals(expectedS, actualS);

    boolean expected = true;
    boolean actual = SnfUtils.nullCheck(null, Boolean.TRUE);
    Assertions.assertEquals(expected, actual);

    expected = false;
    actual = SnfUtils.nullCheck("", Boolean.TRUE, Boolean.FALSE);
    Assertions.assertEquals(expected, actual);
  }



  @Test
  public void testOrderedSet() {
    List<Integer> input = Arrays.asList(1, 5, 6, 8, 8, 9, 100, 100);
    Set<Integer> expected = Collections.unmodifiableSortedSet(new TreeSet<>(input));
    Set<Integer> actual = SnfUtils.toOrderedSet(input);
    Assertions.assertEquals(expected, actual);

    expected = Collections.unmodifiableSortedSet(new TreeSet<>(input));
    actual = SnfUtils.toOrderedSet(true, input.toArray(new Integer[input.size()]));
    Assertions.assertEquals(expected, actual);

  }


  @Test
  public void testParseDate() {
    int year = 2020;
    int month = Calendar.AUGUST;
    int day = 15;

    String f = "%04d%02d%02d";
    LocalDate expected = LocalDate.of(year, month + 1, day);
    LocalDate actual = SnfUtils.parseDate(DateTimeFormatter.BASIC_ISO_DATE,
        String.format(f, year, month + 1, day));
    Assertions.assertEquals(expected, actual);

    year = 2021;
    month = Calendar.JANUARY;
    day = 1;

    expected = LocalDate.of(year, month + 1, day);
    actual = SnfUtils.parseDate(DateTimeFormatter.BASIC_ISO_DATE,
        String.format(f, year, month + 1, day));
    Assertions.assertEquals(expected, actual);

    expected = null;
    actual = SnfUtils.parseDate(DateTimeFormatter.BASIC_ISO_DATE,
        String.format("", year, month + 13, 32));
    Assertions.assertEquals(expected, actual);
  }


  @Test
  public void testNullCheck2() {
    String input = " hello world \t\n";
    final String expected = "good";
    String actual = SnfUtils.nullCheck2(input, () -> null, () -> expected);
    Assertions.assertEquals(expected, actual);

    input = null;
    final String expected2 = "good";
    actual = SnfUtils.nullCheck2(input, () -> expected2, () -> null);
    Assertions.assertEquals(expected2, actual);

  }

  @SuppressWarnings("null")
  @Test
  public void doOrDieTest() {
    try {
      final Integer num = null;
      SnfUtils.doOrDie(() -> num.intValue());
    } catch (Exception ex) {
      Assertions.assertTrue(true);
    }

    Integer expected = 1;
    Integer actual = SnfUtils.doOrDie(() -> expected);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void parseDateTest() {
    String input = "20200929";
    LocalDate expected = LocalDate.of(2020, 9, 29);
    LocalDate actual = SnfUtils.parseDate(DateTimeFormatter.BASIC_ISO_DATE, input);
    Assertions.assertEquals(expected, actual);
  }

  @Test
  public void parseIntTest() {
    Integer expected = 999;
    Integer actual = SnfUtils.parse(expected.toString(), null);
    Assertions.assertEquals(expected, actual);

    expected = null;
    actual = SnfUtils.parse(null, null);
    Assertions.assertEquals(expected, actual);
  }


  @Test
  public void testAnd() {
    Collection<Predicate<Integer>> input = Arrays.asList((num) -> num % 2 == 1, (num) -> num < 100);
    boolean expected = true;
    boolean actual = SnfUtils.and(input).test(3);
    Assertions.assertEquals(expected, actual);

    expected = true;
    actual = SnfUtils.and(input).test(99);
    Assertions.assertEquals(expected, actual);

    expected = false;
    actual = SnfUtils.and(input).test(98);
    Assertions.assertEquals(expected, actual);

    expected = false;
    actual = SnfUtils.and(input).test(101);
    Assertions.assertEquals(expected, actual);


  }

  @Test
  public void testMeetFirstNConditions() {
    for (int i = 0; i < 5; i++) {
      List<Supplier<Boolean>> conditions = new ArrayList<>();
      for (int t = 0; t < i; t++) {
        conditions.add(SnfUtils.of(true));
      }
      for (int j = i; j < 5; j++) {
        conditions.add(SnfUtils.of(false));
      }

      for (int test = 0; test < 5; test++) {
        boolean expected = test <= i;
        boolean actual = SnfUtils.meetFirstNConditions(conditions, test);
        Assertions.assertEquals(expected, actual);
      }
    }

  }

  @Test
  public void testTrimICDCodes() {
    Assertions.assertEquals("A207", SnfUtils.trimICDCodes("A20.7^^"));
    Assertions.assertEquals("J122", SnfUtils.trimICDCodes("J12.2^^^"));
    Assertions.assertEquals("J122", SnfUtils.trimICDCodes("J122"));
  }
}
